<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2024 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

defined('_JEXEC') or die();

use DigitalPeak\ThinHTTP;
use DPCalendar\Helper\DPCalendarHelper;
use Joomla\CMS\Cache\Controller\OutputController;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Session\Session;

class DPCalendarControllerVersion extends BaseController
{
	public function check(): void
	{
		Session::checkToken() or jexit('Invalid token');

		try {
			// Get the current version
			$currentVersion = $this->input->get('DPCALENDAR_VERSION', '');

			/** @var OutputController $cache */
			$cache = Factory::getCache('com_dpcalendar_version', 'output');
			$cache->setCaching(true);
			$cache->setLifeTime(3600);

			if ($cache->contains($currentVersion)) {
				DPCalendarHelper::sendMessage('', false, ['version' => $cache->get($currentVersion)]);

				return;
			}

			// Ignore dev
			if ($currentVersion === 'DP_DEPLOY_VERSION') {
				$cache->store('0', $currentVersion);
				DPCalendarHelper::sendMessage('', false, ['version' => 0]);

				return;
			}

			$url = 'https://cdn.digital-peak.com/update/stream.php?id=19';
			if ($id = DPCalendarHelper::getComponentParameter('downloadid')) {
				$url .= '&dlid=' . $id;
			}

			// Get the current data
			$response = (new ThinHTTP())->get(
				$url . '&j=' . JVERSION . '&p=' . phpversion() . '&m=' . Factory::getDbo()->getVersion() . '&v=' . $currentVersion
			);

			// Check for errors
			if ($response->dp->info->http_code !== 200) {
				$cache->store('0', $currentVersion);
				DPCalendarHelper::sendMessage('', false, ['version' => 0]);

				return;
			}

			// Create the XML
			$xml = new SimpleXMLElement($response->dp->body);
			if (empty($xml->update)) {
				$cache->store('0', $currentVersion);
				DPCalendarHelper::sendMessage('', false, ['version' => 0]);

				return;
			}

			// The first element contains the most recent version
			$update = $xml->update[0];

			// When there is a mismatch, trigger the updates fetch
			if ((string)$update->version !== $currentVersion) {
				BaseDatabaseModel::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_installer/models', 'InstallerModel');
				$model = BaseDatabaseModel::getInstance('Update', 'InstallerModel', ['ignore_request' => true]);
				$model->purge();
				$model->findUpdates(0, 300);
			}

			$cache->store((string)$update->version, $currentVersion);

			// Send the version
			DPCalendarHelper::sendMessage('', false, ['version' => (string)$update->version]);
		} catch (Throwable $e) {
			if ($cache instanceof OutputController) {
				$cache->store('0', $currentVersion);
			}

			DPCalendarHelper::sendMessage('', false, ['version' => 0]);
		}
	}
}
