<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2014 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\DPCalendar\JCalPro\Extension;

\defined('_JEXEC') or die();

use Alledia\Jcalpro\Helper\Helper;
use DigitalPeak\Component\DPCalendar\Administrator\Plugin\DPCalendarPlugin;
use DigitalPeak\Plugin\DPCalendar\JCalPro\Model\EventsModel;
use Joomla\CMS\Categories\Categories;
use Joomla\CMS\Categories\CategoryNode;
use Joomla\CMS\Date\Date;
use Joomla\Event\DispatcherInterface;
use Joomla\Registry\Registry;

class JCalPro extends DPCalendarPlugin
{
	protected string $identifier = 'jc';

	private ?CategoryNode $root = null;

	public function __construct(DispatcherInterface $dispatcher, array $config = [])
	{
		parent::__construct($dispatcher, $config);

		// @phpstan-ignore-next-line
		$root = Categories::getInstance('JCalPro');
		if ($root instanceof Categories) {
			$this->root = $root->get('root');
		}
	}

	protected function getContent(string $calendarId, ?Date $startDate = null, ?Date $endDate = null, ?Registry $options = null): string
	{
		$model = new EventsModel();

		$model->getState();
		$model->setState('filter.catid', $calendarId);
		$model->setState('filter.start_date', $startDate);
		$model->setState('filter.end_date', $endDate);
		$model->setState('list.limit', 0);

		$events = $model->getItems();

		$text   = [];
		$text[] = 'BEGIN:VCALENDAR';
		if (!empty($events)) {
			require_once JPATH_ADMINISTRATOR . '/components/com_jcalpro/include.php';

			foreach ($events as $e) {
				if (empty($e)) {
					continue;
				}

				$text[] = 'BEGIN:VEVENT';
				$text[] = 'UID:' . md5($e->id . 'JcalPro');
				$text[] = 'CATEGORIES:' . $this->params->get('title-' . $calendarId, 'Default');
				$text[] = 'SUMMARY:' . $e->title;
				$text[] = 'DESCRIPTION:' . $this->replaceNl($e->description);

				$text[] = 'X-COLOR:' . str_replace('#', '', (string)$e->color);
				$text[] = 'X-LANGUAGE:' . $e->language;

				if (!empty($e->location_data)) {
					$text[] = 'LOCATION:' . $this->replaceNl($e->location_data->address);
					if (!empty($e->location_data->latitude) && !empty($e->location_data->longitude)) {
						$text[] = 'GEO:' . $e->location_data->latitude . ';' . $e->location_data->longitude;
					}
				}

				$duration_type = (int)$e->duration_type;
				switch ($duration_type) {
					// All day event
					case Helper::EVENT_DURATION_ALL:
						$text[] = 'DTSTART;VALUE=DATE:' . $e->utc_datetime->format('Ymd');
						$text[] = 'DTEND;VALUE=DATE:' . $e->utc_datetime->format('Ymd');
						break;

						// No end date
					case Helper::EVENT_DURATION_NONE:
						$text[] = 'DTSTART:' . $e->utc_datetime->format('Ymd\THis\Z');
						$e->utc_datetime->modify('+1 hour');
						$text[] = 'DTEND:' . $e->utc_datetime->format('Ymd\THis\Z');
						break;

						// Start and end date
					case Helper::EVENT_DURATION_DATE:
					default:
						$text[] = 'DTSTART:' . $e->utc_datetime->format('Ymd\THis\Z');
						if (isset($e->utc_end_datetime) && !empty($e->utc_end_datetime)) {
							$text[] = 'DTEND:' . $e->utc_end_datetime->format('Ymd\THis\Z');
						} else {
							$tmp = clone $e->utc_datetime;
							$tmp->setTime(23, 59, 00);
							$text[] = 'DTEND:' . $tmp->format('Ymd\THis\Z');
						}
				}
				$text[] = 'END:VEVENT';
			}
		}
		$text[] = 'END:VCALENDAR';

		return implode(PHP_EOL, $text);
	}

	protected function fetchCalendars(array $calendarIds = []): array
	{
		$calendars = [];
		if ($this->root == null) {
			return $calendars;
		}

		foreach ($this->root->getChildren(true) as $calendar) {
			if ($calendarIds !== [] && !\in_array($calendar->id, $calendarIds)) {
				continue;
			}

			$calendars[] = $this->createCalendar((string)$calendar->id, $calendar->title, $calendar->description ?: '');
		}

		return $calendars;
	}
}
