<?php
/**
 * @package   DPCalendar
 * @copyright Copyright (C) 2025 Digital Peak GmbH. <https://www.digital-peak.com>
 * @license   https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 */

namespace DigitalPeak\Plugin\DPCalendar\Eventbooking\Extension;

\defined('_JEXEC') or die();

use DigitalPeak\Component\DPCalendar\Administrator\Helper\DPCalendarHelper;
use DigitalPeak\Component\DPCalendar\Administrator\Plugin\DPCalendarPlugin;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Filter\InputFilter;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Registry\Registry;

class Eventbooking extends DPCalendarPlugin
{
	use DatabaseAwareTrait;

	protected string $identifier = 'ebo';

	protected function getContent(string $calendarId, ?Date $startDate = null, ?Date $endDate = null, ?Registry $options = null): string
	{
		if (!is_dir(JPATH_ADMINISTRATOR . '/components/com_eventbooking')) {
			return '';
		}

		$app = $this->getApplication();
		if (!$app instanceof CMSApplicationInterface) {
			return '';
		}

		if (!$options instanceof Registry) {
			$options = new Registry();
		}

		$db = $this->getDatabase();
		if ($startDate == null) {
			// Default to now
			$startDate = DPCalendarHelper::getDate();
		}

		if ($endDate == null) {
			// Default to max end
			$endDate = DPCalendarHelper::getDate(date('Y-m-d H:i:s', PHP_INT_MAX));
		}

		$startDate = $db->quote($startDate);
		$endDate   = $db->quote($endDate);

		$query = $db->getQuery(true);
		$query->select("DISTINCT e.*,concat(l.lat,';',l.long) as location,l.name as location_title");
		$query->from('#__eb_events e');
		$query->where('e.main_category_id = ' . (int)$calendarId);
		$query->where('e.published = 1');
		$query->join('left', '#__eb_locations l on e.location_id = l.id');

		$dateCondition = 'e.event_date  >= ' . $startDate;
		if ($endDate) {
			$dateCondition = '(e.event_end_date between ' . $startDate . ' and ' . $endDate . ' or e.event_date between ' . $startDate . ' and ' . $endDate .
				' or (e.event_date < ' . $startDate . ' and e.event_end_date > ' . $endDate . '))';
		}
		$query->where($dateCondition);

		if (!$options->get('expand')) {
			$query->where('e.parent_id = 0');
		}

		$db->setQuery($query);
		$rows = $db->loadObjectList();

		$text   = [];
		$text[] = 'BEGIN:VCALENDAR';
		foreach ($rows as $event) {
			// For now ignore instances
			if ($event->parent_id != 0) {
				continue;
			}

			$text[] = 'BEGIN:VEVENT';
			$text[] = 'UID:' . md5($event->id . 'Eventbooking');
			$text[] = 'SUMMARY:' . $event->title;
			$text[] = 'DESCRIPTION:' . InputFilter::getInstance()->clean(preg_replace('/\r\n?/', "\N", (string)$event->description));
			$text[] = 'X-ALT-DESC;FMTTYPE=text/html:' . preg_replace('/\r\n?/', "", (string)$event->description);
			$text[] = 'URL:' . $event->event_detail_url;
			$text[] = 'CREATED:' . DPCalendarHelper::getDate($event->created_date)->format('Ymd\THis\Z');

			if ($event->location) {
				$text[] = 'LOCATION:' . $event->location_title;
				$text[] = 'GEO:' . $event->location;
			}

			$allDay = str_contains((string)$event->event_date, ' 00:00:00') && str_contains((string)$event->event_end_date, ' 00:00:00');

			$start  = DPCalendarHelper::getDate($event->event_date, $allDay);
			$text[] = $allDay ? 'DTSTART;VALUE=DATE:' . $start->format('Ymd') : 'DTSTART:' . $start->format('Ymd\THis\Z');

			$end    = DPCalendarHelper::getDate($event->event_end_date !== '0000-00-00 00:00:00' ? $event->event_end_date : (clone $start)->modify('+2 hours'), $allDay);
			$text[] = $allDay ? 'DTEND;VALUE=DATE:' . $end->format('Ymd') : 'DTEND:' . $end->format('Ymd\THis\Z');
			$text[] = 'X-SHOW-END-TIME:' . ($event->event_end_date !== '0000-00-00 00:00:00' ? '1' : '0');

			if ($event->recurring_type != 0) {
				$event->original_id = -1;
				$rrule              = '';
				switch ($event->recurring_type) {
					case 1:
						$rrule = 'FREQ=DAILY';
						break;
					case 2:
						$map   = ['SU', 'MO', 'TU', 'WE', 'TH', 'FR', 'SA'];
						$rrule = 'FREQ=WEEKLY';
						$rrule .= ';BYDAY=' . implode(',', array_map(fn ($day): string => $map[$day], explode(',', (string)$event->weekdays)));
						break;
					case 3:
						$rrule = 'FREQ=MONTHLY;BYMONTHDAY=' . $event->monthdays;
						break;
					case 4:
						$map    = ['first' => 1, 'second' => 2, 'third' => 3, 'fourth' => 4, 'fifth' => 5, 'last' => -1];
						$params = new Registry($event->params);
						$rrule  = 'FREQ=MONTHLY';
						$rrule .= ';BYDAY=' . $map[$params->get('week_in_month', 'first')] . strtoupper(substr((string)$params->get('day_of_week', 'SUN'), 0, 2));
						break;
				}

				if ($event->recurring_end_date !== '0000-00-00 00:00:00') {
					$rrule .= ';UNTIL=' . $event->recurring_end_date;
				}

				if ($event->recurring_occurrencies) {
					$rrule .= ';COUNT=' . $event->recurring_occurrencies;
				}

				if ($event->recurring_frequency) {
					$rrule .= ';INTERVAL=' . $event->recurring_frequency;
				}

				$text[] = 'RRULE:' . $rrule;
			}

			$text[] = 'X-IMAGE-INTRO:' . $event->image;
			$text[] = 'X-IMAGE-INTRO-ALT:' . $event->image_alt;
			$text[] = 'X-IMAGE-FULL:' . $event->image;
			$text[] = 'X-IMAGE-FULL-ALT:' . $event->image_alt;
			$text[] = 'X-ACCESS:' . $event->access;
			$text[] = 'X-ALIAS:' . $event->alias;
			$text[] = 'X-LANGUAGE:' . $event->language;
			$text[] = 'X-PUBLISH-UP:' . $event->publish_up;
			$text[] = 'X-PUBLISH-DOWN:' . $event->publish_down;

			$text[] = 'END:VEVENT';
		}
		$text[] = 'END:VCALENDAR';

		return implode(PHP_EOL, $text);
	}

	protected function fetchCalendars(array $calendarIds = []): array
	{
		if (!is_dir(JPATH_ADMINISTRATOR . '/components/com_eventbooking')) {
			return [];
		}

		require_once JPATH_ADMINISTRATOR . '/components/com_eventbooking/libraries/rad/bootstrap.php';

		// @phpstan-ignore-next-line
		BaseDatabaseModel::addIncludePath(JPATH_SITE . '/components/com_eventbooking/model');

		/** @var \EventbookingModelCategories $model */
		$model = \RADModel::getInstance('Categories', 'EventbookingModel', ['table_prefix' => '#__eb_']);
		$model->setParams(new Registry(['category_ids' => $calendarIds]));

		$calendars = [];
		foreach ($model->getData() as $category) {
			$calendars[] = $this->createCalendar((string)$category->id, $category->title ?? $category->name, $category->description);
		}

		return $calendars;
	}
}
